/*
 * Copyright 2007 David F. Elliott.  All rights reserved.
 */

#include "libsaio.h"
#include "bootstruct.h" /* for bootArgs */
#include <pexpert/i386/efi.h>
#include "fake_efi.h"
#include "efi_tables.h"
#include "SMBIOS.h"

/*
Modern Darwin kernels require some amount of EFI because Apple machines all
have EFI.  Modifying the kernel source to not require EFI is of course
possible but would have to be maintained as a separate patch because it is
unlikely that Apple wishes to add legacy support to their kernel.

As you can see from the Apple-supplied code in bootstruct.c, it seems that
the intention was clearly to modify this booter to provide EFI-like structures
to the kernel rather than modifying the kernel to handle non-EFI stuff.  This
makes a lot of sense from an engineering point of view as it means the kernel
for the as yet unreleased EFI-only Macs could still be booted by the non-EFI
DTK systems so long as the kernel checked to ensure the boot tables were
filled in appropriately.  Modern xnu requires a system table and a runtime
services table and performs no checks whatsoever to ensure the pointers to
these tables are non-NULL.  Therefore, any modern xnu kernel will page fault
early on in the boot process if the system table pointer is zero.

Even before that happens, the tsc_init function in modern xnu requires the FSB
Frequency to be a property in the /efi/platform node of the device tree or else
it panics the bootstrap process very early on.

As of this writing, the current implementation found here is good enough
to make the currently available xnu kernel boot without modification on a
system with an appropriate processor.  With a minor source modification to
the tsc_init function to remove the explicit check for Core or Core 2
processors the kernel can be made to boot on other processors so long as
the code can be executed by the processor and the machine contains the
necessary hardware.
*/


/*==========================================================================
 * Utility function to make a device tree string from an EFI_GUID
 */

static inline char * mallocStringForGuid(EFI_GUID const *pGuid)
{
    char *string = malloc(37);
    efi_guid_unparse_upper(pGuid, string);
    return string;
}


/*==========================================================================
 * Fake EFI implementation
 */

/* Identify ourselves as the EFI firmware vendor */
static EFI_CHAR16 const FIRMWARE_VENDOR[] = {'D','a','r','w','i','n','_','b','o','o','t', 0};
static EFI_UINT32 const FIRMWARE_REVISION = 132; /* FIXME: Find a constant for this. */

/* Just a ret instruction */
static uint8_t const VOIDRET_INSTRUCTIONS[] = {0xc3};
/* movl $0x80000003,%eax; ret */
static uint8_t const UNSUPPORTEDRET_INSTRUCTIONS[] = {0xb8, 0x03, 0x00, 0x00, 0x80, 0xc3};

/* Set up space for up to 10 configuration table entries */
#define MAX_CONFIGURATION_TABLE_ENTRIES 10

/* We use the fake_efi_pages struct so that we only need to do one kernel
 * memory allocation for all needed EFI data.  Otherwise, small allocations
 * like the FIRMWARE_VENDOR string would take up an entire page.
 * NOTE WELL: Do NOT assume this struct has any particular layout within itself.
 * It is absolutely not intended to be publicly exposed anywhere
 * We say pages (plural) although right now we are well within the 1 page size
 * and probably will stay that way.
 */
struct fake_efi_pages
{
    EFI_SYSTEM_TABLE efiSystemTable;
    EFI_RUNTIME_SERVICES efiRuntimeServices;
    EFI_CONFIGURATION_TABLE efiConfigurationTable[MAX_CONFIGURATION_TABLE_ENTRIES];
    EFI_CHAR16 firmwareVendor[sizeof(FIRMWARE_VENDOR)/sizeof(EFI_CHAR16)];
    uint8_t voidret_instructions[sizeof(VOIDRET_INSTRUCTIONS)/sizeof(uint8_t)];
    uint8_t unsupportedret_instructions[sizeof(UNSUPPORTEDRET_INSTRUCTIONS)/sizeof(uint8_t)];
};

EFI_SYSTEM_TABLE *gST = NULL;
Node *gEfiConfigurationTableNode = NULL;

static EFI_STATUS addConfigurationTable(EFI_GUID const *pGuid, void *table, char const *alias)
{
    EFI_UINTN i = gST->NumberOfTableEntries;
    /* We only do adds, not modifications and deletes like InstallConfigurationTable */
    if(i >= MAX_CONFIGURATION_TABLE_ENTRIES)
        stop("Ran out of space for configuration tables.  Increase the reserved size in the code.\n");

    if(pGuid == NULL)
        return EFI_INVALID_PARAMETER;

    if(table != NULL)
    {
        gST->ConfigurationTable[i].VendorGuid = *pGuid;
        gST->ConfigurationTable[i].VendorTable = table;

        ++gST->NumberOfTableEntries;

        Node *tableNode = DT__AddChild(gEfiConfigurationTableNode, mallocStringForGuid(pGuid));
        /* Use the pointer to the GUID we just stuffed into the system table */
        DT__AddProperty(tableNode, "guid", sizeof(EFI_GUID),&gST->ConfigurationTable[i].VendorGuid);
	/* The "table" property is the 32-bit (in our implementation) physical address of the table */
        DT__AddProperty(tableNode, "table", sizeof(void*), &gST->ConfigurationTable[i].VendorTable);
        /* Assume the alias pointer is a global or static piece of data */
        if(alias != NULL)
            DT__AddProperty(tableNode, "alias", strlen(alias)+1, (char*)alias);
        return EFI_SUCCESS;
    }
    return EFI_UNSUPPORTED;
}

static inline void fixupEfiSystemTableCRC32(EFI_SYSTEM_TABLE *efiSystemTable)
{
    efiSystemTable->Hdr.CRC32 = 0;
    efiSystemTable->Hdr.CRC32 = crc32(0L, efiSystemTable, efiSystemTable->Hdr.HeaderSize);
}

/*
What we do here is simply allocate a fake EFI system table and a fake EFI
runtime services table.

Because we build against modern headers with kBootArgsRevision 4 we
also take care to set efiMode = 32.
*/
void
setupEfiTables(void)
{
    struct fake_efi_pages *fakeEfiPages= (struct fake_efi_pages*)AllocateKernelMemory(sizeof(struct fake_efi_pages));

    /* Zero out all the tables in case fields are added later */
    bzero(fakeEfiPages, sizeof(struct fake_efi_pages));

    /* --------------------------------------------------------------------
     * Initialize some machine code that will return EFI_UNSUPPORTED for
     * functions returning int and simply return for void functions.
     */
    memcpy(fakeEfiPages->voidret_instructions, VOIDRET_INSTRUCTIONS, sizeof(VOIDRET_INSTRUCTIONS));
    memcpy(fakeEfiPages->unsupportedret_instructions, UNSUPPORTEDRET_INSTRUCTIONS, sizeof(UNSUPPORTEDRET_INSTRUCTIONS));

    /* -------------------------------------------------------------------- */
    /* System table */
    EFI_SYSTEM_TABLE *efiSystemTable = gST = &fakeEfiPages->efiSystemTable;
    efiSystemTable->Hdr.Signature = EFI_SYSTEM_TABLE_SIGNATURE;
    efiSystemTable->Hdr.Revision = EFI_SYSTEM_TABLE_REVISION;
    efiSystemTable->Hdr.HeaderSize = sizeof(EFI_SYSTEM_TABLE);
    efiSystemTable->Hdr.CRC32 = 0; /* Initialize to zero and then do CRC32 */
    efiSystemTable->Hdr.Reserved = 0;

    efiSystemTable->FirmwareVendor = fakeEfiPages->firmwareVendor;
    memcpy(efiSystemTable->FirmwareVendor, FIRMWARE_VENDOR, sizeof(FIRMWARE_VENDOR));
    efiSystemTable->FirmwareRevision = FIRMWARE_REVISION;

    /* XXX: We may need to have basic implementations of ConIn/ConOut/StdErr */
    /* The EFI spec states that all handles are invalid after boot services have been
     * exited so we can probably get by with leaving the handles as zero. */
    efiSystemTable->ConsoleInHandle = 0;
    efiSystemTable->ConIn = 0;

    efiSystemTable->ConsoleOutHandle = 0;
    efiSystemTable->ConOut = 0;

    efiSystemTable->StandardErrorHandle = 0;
    efiSystemTable->StdErr = 0;

    efiSystemTable->RuntimeServices = &fakeEfiPages->efiRuntimeServices;
    /* According to the EFI spec, BootServices aren't valid after the
     * boot process is exited so we can probably do without it.
     * Apple didn't provide a definition for it in pexpert/i386/efi.h
     * so I'm guessing they don't use it.
    */
    efiSystemTable->BootServices = 0;

    efiSystemTable->NumberOfTableEntries = 0;
    efiSystemTable->ConfigurationTable = fakeEfiPages->efiConfigurationTable;


    /* We're done.  Now CRC32 the thing so the kernel will accept it */
    fixupEfiSystemTableCRC32(efiSystemTable);

    /* -------------------------------------------------------------------- */
    /* Runtime services */
    EFI_RUNTIME_SERVICES *efiRuntimeServices = &fakeEfiPages->efiRuntimeServices;
    efiRuntimeServices->Hdr.Signature = EFI_RUNTIME_SERVICES_SIGNATURE;
    efiRuntimeServices->Hdr.Revision = EFI_RUNTIME_SERVICES_REVISION;
    efiRuntimeServices->Hdr.HeaderSize = sizeof(EFI_RUNTIME_SERVICES);
    efiRuntimeServices->Hdr.CRC32 = 0;
    efiRuntimeServices->Hdr.Reserved = 0;

    /* There are a number of function pointers in the efiRuntimeServices table.
     * These are the Foundation (e.g. core) services and are expected to be present on
     * all EFI-compliant machines.  Some kernel extensions (notably AppleEFIRuntime)
     * will call these without checking to see if they are null.
     *
     * We don't really feel like doing an EFI implementation in the bootloader
     * but it is nice if we can at least prevent a complete crash by
     * at least providing some sort of implementation until one can be provided
     * nicely in a kext.
     */
    void (*voidret_fp)() = (void*)fakeEfiPages->voidret_instructions;
    void (*unsupportedret_fp)() = (void*)fakeEfiPages->unsupportedret_instructions;
    efiRuntimeServices->GetTime = (void*)unsupportedret_fp;
    efiRuntimeServices->SetTime = (void*)unsupportedret_fp;
    efiRuntimeServices->GetWakeupTime = (void*)unsupportedret_fp;
    efiRuntimeServices->SetWakeupTime = (void*)unsupportedret_fp;
    efiRuntimeServices->SetVirtualAddressMap = (void*)unsupportedret_fp;
    efiRuntimeServices->ConvertPointer = (void*)unsupportedret_fp;
    efiRuntimeServices->GetVariable = (void*)unsupportedret_fp;
    efiRuntimeServices->GetNextVariableName = (void*)unsupportedret_fp;
    efiRuntimeServices->SetVariable = (void*)unsupportedret_fp;
    efiRuntimeServices->GetNextHighMonotonicCount = (void*)unsupportedret_fp;
    efiRuntimeServices->ResetSystem = (void*)voidret_fp;

    /* We're done.  Now CRC32 the thing so the kernel will accept it */
    efiRuntimeServices->Hdr.CRC32 = crc32(0L, efiRuntimeServices, efiRuntimeServices->Hdr.HeaderSize);


    /* -------------------------------------------------------------------- */
    /* Finish filling in the rest of the boot args that we need. */
    bootArgs->efiSystemTable = (uint32_t)efiSystemTable;
    bootArgs->efiMode = kBootArgsEfiMode32;

    /* The bootArgs structure as a whole is bzero'd so we don't need to fill in
     * things like efiRuntimeServices* and what not.
     *
     * In fact, the only code that seems to use that is the hibernate code so it
     * knows not to save the pages.  It even checks to make sure its nonzero.
     */
}

/*
In addition to the EFI tables there is also the EFI device tree node.
In particular, we need /efi/platform to have an FSBFrequency key. Without it,
the tsc_init function will panic very early on in kernel startup, before
the console is available.
*/

/* Decimal powers: */
#define kilo (1000ULL)
#define Mega (kilo * kilo)
#define Giga (kilo * Mega)
#define Tera (kilo * Giga)
#define Peta (kilo * Tera)

/*==========================================================================
 * FSB Frequency detection
 */
// DFE: rdtsc64 from xnu (obvious implementation)
static inline uint64_t rdtsc64(void)
{
    uint64_t ret;
    __asm__ volatile("rdtsc" : "=A" (ret));
    return ret;
}

// DFE: enable_PIT2 and disable_PIT2 come from older xnu

/*
 * Enable or disable timer 2.
 * Port 0x61 controls timer 2:
 *   bit 0 gates the clock,
 *   bit 1 gates output to speaker.
 */
inline static void
enable_PIT2(void)
{
    /* Enable gate, disable speaker */
    __asm__ volatile(
        " inb   $0x61,%%al      \n\t"
        " and   $0xFC,%%al       \n\t"  /* & ~0x03 */
        " or    $1,%%al         \n\t"
        " outb  %%al,$0x61      \n\t"
        : : : "%al" );
}

inline static void
disable_PIT2(void)
{
    /* Disable gate and output to speaker */
    __asm__ volatile(
        " inb   $0x61,%%al      \n\t"
        " and   $0xFC,%%al      \n\t"	/* & ~0x03 */
        " outb  %%al,$0x61      \n\t"
        : : : "%al" );
}

// DFE: set_PIT2_mode0, poll_PIT2_gate, and measure_tsc_frequency are
// roughly based on Linux code

/* Set the 8254 channel 2 to mode 0 with the specified value.
   In mode 0, the counter will initially set its gate low when the
   timer expires.  For this to be useful, you ought to set it high
   before calling this function.  The enable_PIT2 function does this.
 */
static inline void set_PIT2_mode0(uint16_t value)
{
    __asm__ volatile(
        " movb  $0xB0,%%al      \n\t"
        " outb	%%al,$0x43	\n\t"
        " movb	%%dl,%%al	\n\t"
        " outb	%%al,$0x42	\n\t"
        " movb	%%dh,%%al	\n\t"
        " outb	%%al,$0x42"
        : : "d"(value) /*: no clobber */ );
}

/* Returns the number of times the loop ran before the PIT2 signaled */
static inline unsigned long poll_PIT2_gate(void)
{
    unsigned long count = 0;
    unsigned char nmi_sc_val;
    do {
        ++count;
        __asm__ volatile(
            "inb	$0x61,%0"
        : "=q"(nmi_sc_val) /*:*/ /* no input */ /*:*/ /* no clobber */);
    } while( (nmi_sc_val & 0x20) == 0);
    return count;
}

// DFE: This constant comes from older xnu:
#define CLKNUM		1193182		/* formerly 1193167 */

// DFE: These two constants come from Linux except CLOCK_TICK_RATE replaced with CLKNUM
#define CALIBRATE_TIME_MSEC 30 /* 30 msecs */
#define CALIBRATE_LATCH	\
	((CLKNUM * CALIBRATE_TIME_MSEC + 1000/2)/1000)

/*
 * Measures the TSC frequency in Hz (64-bit) using the ACPI PM timer
 */
uint64_t measure_tsc_frequency(void);
uint64_t measure_tsc_frequency(void)
{
    uint64_t tscStart;
    uint64_t tscEnd;
    uint64_t tscDelta = 0xffffffffffffffffULL;
    unsigned long pollCount;
    uint64_t retval = 0;
    int i;

    /* Time how many TSC ticks elapse in 30 msec using the 8254 PIT
     * counter 2.  We run this loop 3 times to make sure the cache
     * is hot and we take the minimum delta from all of the runs.
     * That is to say that we're biased towards measuring the minimum
     * number of TSC ticks that occur while waiting for the timer to
     * expire.  That theoretically helps avoid inconsistencies when
     * running under a VM if the TSC is not virtualized and the host
     * steals time.  The TSC is normally virtualized for VMware.
     */
    for(i = 0; i < 3; ++i)
    {
        enable_PIT2();
        set_PIT2_mode0(CALIBRATE_LATCH);
        tscStart = rdtsc64();
        pollCount = poll_PIT2_gate();
        tscEnd = rdtsc64();
        /* The poll loop must have run at least a few times for accuracy */
        if(pollCount <= 1)
            continue;
        /* The TSC must increment at LEAST once every millisecond.  We
         * should have waited exactly 30 msec so the TSC delta should
         * be >= 30.  Anything less and the processor is way too slow.
         */
        if((tscEnd - tscStart) <= CALIBRATE_TIME_MSEC)
            continue;
        // tscDelta = min(tscDelta, (tscEnd - tscStart))
        if( (tscEnd - tscStart) < tscDelta )
            tscDelta = tscEnd - tscStart;
    }
    /* tscDelta is now the least number of TSC ticks the processor made in
     * a timespan of 0.03 s (e.g. 30 milliseconds)
     * Linux thus divides by 30 which gives the answer in kiloHertz because
     * 1 / ms = kHz.  But we're xnu and most of the rest of the code uses
     * Hz so we need to convert our milliseconds to seconds.  Since we're
     * dividing by the milliseconds, we simply multiply by 1000.
     */

    /* Unlike linux, we're not limited to 32-bit, but we do need to take care
     * that we're going to multiply by 1000 first so we do need at least some
     * arithmetic headroom.  For now, 32-bit should be enough.
     * Also unlike Linux, our compiler can do 64-bit integer arithmetic.
     */
    if(tscDelta > (1ULL<<32))
        retval = 0;
    else
    {
        retval = tscDelta * 1000 / 30;
    }
    disable_PIT2();
    return retval;
}

static inline uint64_t rdmsr64(uint32_t msr)
{
	uint64_t ret;
	__asm__ volatile("rdmsr" : "=A" (ret) : "c" (msr));
	return ret;
}

#define MSR_IA32_PERF_STS		0x198

#define bit(n)		(1ULL << (n))
#define bitmask(h,l)	((bit(h)|(bit(h)-1)) & ~(bit(l)-1))
#define bitfield(x,h,l)	(((x) & bitmask(h,l)) >> l)

uint64_t fsbFrequency = 0;

void determineFsbFrequency()
{
    const char *val; // value
    int cnt; // length of value

    if(getValueForKey("fsbmhz", &val, &cnt))
    {
        uint32_t fsbmhz;
        char valcpy[10];
        bool gotvalue = (cnt < 10);
        if(gotvalue)
        {
            // Force NUL-term by making a copy with strlcpy
            strlcpy(valcpy, val, cnt+1);
            char *endval;
            fsbmhz = strtoul(valcpy, &endval, 10);
            gotvalue = (valcpy != endval && *endval == '\0');
        }
        if(gotvalue)
        {
            switch(fsbmhz)
            {
            case 33:
                fsbFrequency =  33333333;
                break;
            case 66:
                fsbFrequency =  66666667;
                break;
            case 133:
                fsbFrequency = 133333333;
                break;
            case 166:
                fsbFrequency = 166666667;
                break;
            default:
                printf("Strange fsbmhz=%d specified.. assuming you mean it literally.\n", fsbmhz);
                /* Fall through */
            case 100:
            case 200:
                fsbFrequency = fsbmhz * Mega;
            }
        }
        else
            fsbFrequency = 0;
    }
    else
        fsbFrequency = 0;

    if(fsbFrequency != 0)
    {
        printf("Using specified FSB Frequency %d.%06dMHz\n", (uint32_t)(fsbFrequency/Mega), (uint32_t)(fsbFrequency % Mega));
        sleep(1);
        return;
    }
    printf("Attempting to determine CPU Multiplier.  If system reboots, RDMSR 198h causes a fault.\n");
    sleep(1);
    uint64_t prfsts = rdmsr64(MSR_IA32_PERF_STS);
    uint32_t tscGranularity = bitfield(prfsts, 44, 40);
    printf("Determined CPU:FSB multiplier to be %d\n", tscGranularity);
    if(tscGranularity == 0)
    {
        printf("CPU says its multiplier is 0 which makes no sense. The kernel as shipped by\n");
        printf("Apple will not support this and will cause the machine to reboot immediately.\n");
        printf("Press 'y' to continue or use Ctrl+Alt+Delete to reboot\n");
        while(getc() != 'y')
            ;
        fsbFrequency = 0;
        return;
    }
    printf("Measuring TSC frequency (e.g. CPU speed)\n");
    uint64_t measuredTscFrequency = measure_tsc_frequency();
    printf("CPU runs at %d.%06d MHz\n", (uint32_t)(measuredTscFrequency / Mega), (uint32_t)(measuredTscFrequency % Mega));
    fsbFrequency = measuredTscFrequency / tscGranularity;
    printf("BUS must therefore run at %d.%06d MHz\n", (uint32_t)(fsbFrequency / Mega), (uint32_t)(fsbFrequency % Mega));
    sleep(1);
}

void printFrequencyInfo()
{
    printf("\n");
    uint64_t measuredTscFrequency = measure_tsc_frequency();
    printf("CPU runs at %d.%06d MHz\n", (uint32_t)(measuredTscFrequency / Mega), (uint32_t)(measuredTscFrequency % Mega));
    printf("\n");
    printf("(Press a key to continue...)");
    getc();
}

/* This should be const but DT__AddProperty takes char* */
static char FSB_Frequency_prop[] = "FSBFrequency";


/*==========================================================================
 * SMBIOS
 */

/* From Foundation/Efi/Guid/Smbios/SmBios.h */
/* Modified to wrap Data4 array init with {} */
#define EFI_SMBIOS_TABLE_GUID \
  { \
    0xeb9d2d31, 0x2d88, 0x11d3, {0x9a, 0x16, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d} \
  }

/* From Foundation/Efi/Guid/Smbios/SmBios.c */
EFI_GUID const  gEfiSmbiosTableGuid = EFI_SMBIOS_TABLE_GUID;

#define SMBIOS_RANGE_START      0x000F0000
#define SMBIOS_RANGE_END        0x000FFFFF

/* '_SM_' in little endian: */
#define SMBIOS_ANCHOR_UINT32_LE 0x5f4d535f

static inline void* getAddressOfSmbiosTable()
{
    /* First see if we can even find the damn SMBIOS table
     * The logic here is to start at 0xf0000 and end at 0xfffff iterating 16 bytes at a time looking
     * for the SMBIOS entry-point structure anchor (literal ASCII "_SM_").
     */
    void *smbios_addr = (void*)SMBIOS_RANGE_START;
    for(; (smbios_addr <= (void*)SMBIOS_RANGE_END) && (*(uint32_t*)smbios_addr != SMBIOS_ANCHOR_UINT32_LE); smbios_addr += 16)
        ;
    if(smbios_addr <= (void*)SMBIOS_RANGE_END)
    {
        /* NOTE: The specification does not specifically state what to do in the event of finding an
         * SMBIOS anchor on an invalid table.  It might be better to move this code into the for loop
         * so that searching can continue.
         */
        uint8_t csum = checksum8(smbios_addr, sizeof(struct SMBEntryPoint));
        /* The table already contains the checksum so we merely need to see if its checksum is now zero. */
        if(csum != 0)
        {
            printf("Found SMBIOS anchor but bad table checksum.  Assuming no SMBIOS.\n");
            sleep(5);
            smbios_addr = 0;
        }
    }
    else
    {
        /* If this happens, it's possible that a PnP BIOS call can be done to retrieve the address of the table.
         * The latest versions of the spec state that modern programs should not even attempt to do this. */
        printf("Unable to find SMBIOS table.\n");
        sleep(5);
        smbios_addr = 0;
    }
    return smbios_addr;
}


/*==========================================================================
 * ACPI
 */

#define EFI_ACPI_TABLE_GUID \
  { \
    0xeb9d2d30, 0x2d88, 0x11d3, { 0x9a, 0x16, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d } \
  }

#define EFI_ACPI_20_TABLE_GUID \
  { \
    0x8868e871, 0xe4f1, 0x11d3, { 0xbc, 0x22, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81 } \
  }

// NOTE: ACPI 3.0 uses same GUID as ACPI 2.0

EFI_GUID  gEfiAcpiTableGuid = EFI_ACPI_TABLE_GUID;

EFI_GUID  gEfiAcpi20TableGuid = EFI_ACPI_20_TABLE_GUID;

#define ACPI_RANGE_START    (0x0E0000)
#define ACPI_RANGE_END      (0x0FFFFF)

#define UINT64_LE_FROM_CHARS(a,b,c,d,e,f,g,h) \
    (   ((uint64_t)h << 56) \
    |   ((uint64_t)g << 48) \
    |   ((uint64_t)f << 40) \
    |   ((uint64_t)e << 32) \
    |   ((uint64_t)d << 24) \
    |   ((uint64_t)c << 16) \
    |   ((uint64_t)b <<  8) \
    |   ((uint64_t)a <<  0) \
    )

#define ACPI_SIGNATURE_UINT64_LE UINT64_LE_FROM_CHARS('R','S','D',' ','P','T','R',' ')

/* Per ACPI 3.0a spec */
struct acpi_2_rsdp {
    char            Signature[8];
    uint8_t         Checksum;
    char            OEMID[6];
    uint8_t         Revision;
    uint32_t        RsdtAddress;
    uint32_t        Length;
    uint64_t        XsdtAddress;
    uint8_t         ExtendedChecksum;
    char            Reserved[3];
} __attribute__((packed));

/* Gets the ACPI 1.0 RSDP address */
static void* getAddressOfAcpiTable()
{
    /* TODO: Before searching the BIOS space we are supposed to search the first 1K of the EBDA */

    void *acpi_addr = (void*)ACPI_RANGE_START;
    for(; acpi_addr <= (void*)ACPI_RANGE_END; acpi_addr += 16)
    {
        if(*(uint64_t *)acpi_addr == ACPI_SIGNATURE_UINT64_LE)
        {
            uint8_t csum = checksum8(acpi_addr, 20);
            if(csum == 0)
            {
                // Only return the table if it is a true version 1.0 table (Revision 0)
                if(((struct acpi_2_rsdp*)acpi_addr)->Revision == 0)
                    return acpi_addr;
            }
        }
    }
    return NULL;
}

/* Gets the ACPI 2.0 RSDP address */
static void* getAddressOfAcpi20Table()
{
    /* TODO: Before searching the BIOS space we are supposed to search the first 1K of the EBDA */

    void *acpi_addr = (void*)ACPI_RANGE_START;
    for(; acpi_addr <= (void*)ACPI_RANGE_END; acpi_addr += 16)
    {
        if(*(uint64_t *)acpi_addr == ACPI_SIGNATURE_UINT64_LE)
        {
            uint8_t csum = checksum8(acpi_addr, 20);
            /* Only assume this is a 2.0 or better table if the revision is greater than 0
             * NOTE: ACPI 3.0 spec only seems to say that 1.0 tables have revision 1
             * and that the current revision is 2.. I am going to assume that rev > 0 is 2.0.
             */
            if(csum == 0 && (((struct acpi_2_rsdp*)acpi_addr)->Revision > 0))
            {
                uint8_t csum2 = checksum8(acpi_addr, sizeof(struct acpi_2_rsdp));
                if(csum2 == 0)
                    return acpi_addr;
            }
        }
    }
    return NULL;
}


/*==========================================================================
 * Fake EFI implementation
 */

char const FIRMWARE_REVISION_PROP[] = "firmware-revision";
char const FIRMWARE_ABI_PROP[] = "firmware-abi";
char const FIRMWARE_VENDOR_PROP[] = "firmware-vendor";
char const FIRMWARE_ABI_PROP_VALUE[] = "EFI32";

void
setupEfiDeviceTree(void)
{
    Node *node;
    node = DT__FindNode("/", false);
    if (node == 0) {
        stop("Couldn't get root node");
    }
    /* We could also just do DT__FindNode("/efi/platform", true)
     * But I think eventually we want to fill stuff in the efi node
     * too so we might as well create it so we have a pointer for it too.
    */
    node = DT__AddChild(node, "efi");

    DT__AddProperty(node, FIRMWARE_REVISION_PROP, sizeof(FIRMWARE_REVISION), (EFI_UINT32*)&FIRMWARE_REVISION);
    DT__AddProperty(node, FIRMWARE_ABI_PROP, sizeof(FIRMWARE_ABI_PROP_VALUE), (char*)FIRMWARE_ABI_PROP_VALUE);
    DT__AddProperty(node, FIRMWARE_VENDOR_PROP, sizeof(FIRMWARE_VENDOR), (EFI_CHAR16*)FIRMWARE_VENDOR);

    /* TODO: Fill in other efi properties if necessary */

    /* Set up the /efi/runtime-services table node similar to the way a child node of configuration-table
     * is set up.  That is, name and table properties */
    Node *runtimeServicesNode = DT__AddChild(node, "runtime-services");

    /* The value of the table property is the 32-bit physical address for the RuntimeServices table.
     * Sice the EFI system table already has a pointer to it, we simply use the address of that pointer
     * for the pointer to the property data.  Warning.. DT finalization calls free on that but we're not
     * the only thing to use a non-malloc'd pointer for something in the DT
     */
    DT__AddProperty(runtimeServicesNode, "table", sizeof(void *), &gST->RuntimeServices);

    /* Set up the /efi/configuration-table node which will eventually have several child nodes for
     * all of the configuration tables needed by various kernel extensions.
     */
    gEfiConfigurationTableNode = DT__AddChild(node, "configuration-table");

    /* Now fill in the /efi/platform Node */
    Node *efiPlatformNode = DT__AddChild(node, "platform");

    if(fsbFrequency != 0)
        DT__AddProperty(efiPlatformNode, FSB_Frequency_prop, sizeof(uint64_t), &fsbFrequency);
    /* NOTE WELL: If you do add FSB Frequency detection, make sure to store
     * the value in the fsbFrequency global and not an malloc'd pointer
     * because the DT_AddProperty function does not copy its args.
     */
}

/* Installs all the needed configuration table entries */
void setupEfiConfigurationTable()
{
    addConfigurationTable(&gEfiSmbiosTableGuid, getAddressOfSmbiosTable(), NULL);
    addConfigurationTable(&gEfiAcpiTableGuid, getAddressOfAcpiTable(), "ACPI");
    addConfigurationTable(&gEfiAcpi20TableGuid, getAddressOfAcpi20Table(), "ACPI_20");

    // We've obviously changed the count.. so fix up the CRC32
    fixupEfiSystemTableCRC32(gST);
}

/* Entrypoint from boot.c */
void setupFakeEfi(void)
{
    determineFsbFrequency();
    // Initialize the base table
    setupEfiTables();
    // Initialize the device tree
    setupEfiDeviceTree();
    // Add configuration table entries to both the services table and the device tree
    setupEfiConfigurationTable();
}

